/*
** FCD standard & high compress decoding module
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "binary.h"
#include "fcd.h"

#include "fcd_hi_decompress.h"

#define FCD_DECODE_C
#include "fcd_decode.h"

typedef struct{
	int data;
	int rest;
} flag;

typedef struct{
	int data;
	int rest;
} subflag;

typedef struct{
	char **index;
	int last;
} dictionary;

typedef struct {
	FILE *stream; /* fcd compressed data stream */
	FILE *debug;  /* debug output               */
    
	flag f;       
	subflag sf;
	dictionary dict;
    
	char *data;   /* decoded data buffer        */
	int db_len;   /* dec data buffer length     */
	int db_next;  /* next unused dec buffer     */
    
	char *eb;     /* encoded data buffer        */
	short eb_len; /* enc data buffer length     */
	int eb_pos;   /* enc data buffer position   */
    
} fcd_decoding_buffer;

int fcd_std_unit_decode(FCD *in, cd_image_buffer *out, FILE *debug);
int fcd_hi_unit_decode(FCD *in, cd_image_buffer *out, FILE *debug);
static int get_flag(fcd_decoding_buffer *fdb);
static int get_subflag(fcd_decoding_buffer *fdb, int bits);
static int get_adv_offset(fcd_decoding_buffer *fdb);
static int store_original(fcd_decoding_buffer *fdb, int length);
static int restore_pattern(fcd_decoding_buffer *fdb, int offset, int length);
static void print_flag(int flag, int column, FILE *out);
static int ebgetc(fcd_decoding_buffer *fdb);

int fcd_std_unit_decode(FCD *in, cd_image_buffer *out, FILE *debug)
{
	size_t n;

	fcd_decoding_buffer *fdb;

	int flag, subflag;
	int offset;
	int length;

	fdb = (fcd_decoding_buffer *)calloc(1,sizeof(fcd_decoding_buffer));

	fdb->stream = in->stream;

	fdb->debug = debug;

	fdb->db_len = out->block_size * 16;

	fdb->data = (char *)calloc(fdb->db_len, 1);

	read_le_int16(fdb->stream, &(fdb->eb_len));
	in->pos += 2;

	if(fdb->eb_len & 0x8000){ /* unit is compressed */
		fdb->eb_len &= 0x7FFF;
		fdb->eb = (char *)malloc(fdb->eb_len);
		if((n = fread(fdb->eb, 1, fdb->eb_len, in->stream)) != fdb->eb_len){
			perror(in->path);
			exit(EXIT_FAILURE);
		}
		in->pos += n;
		fdb->eb_pos = 0;
	}else{ /* unit is uncompressed */
		if(fdb->eb_len != 0x7FFF){

			/*********************************************************
			              Fucking FCD spec infomation

			    If a unit length is set to 0x7FFF, the correct
			    value is 0x8000. (already setting, don't touch)

			    In the other case, unit length is a correct value.
			    (probably, I have not seem the case yet)
			*********************************************************/

			fdb->db_len = (unsigned short)fdb->eb_len;

		}
		if((n = fread(fdb->data, 1, fdb->db_len, in->stream)) != fdb->db_len){
			perror(in->path);
			exit(EXIT_FAILURE);
		}
		in->pos += n;
		out->data = fdb->data;
		out->num_of_block = n / out->block_size;
		free(fdb);
		return out->num_of_block;
	}

	fdb->dict.index = (char **)calloc(fdb->db_len, sizeof(char *));
	fdb->dict.last = -1;

	while((flag = get_flag(fdb)) >= 0){
		switch(flag){
		case 0x0: /* restore 3 bytes pattern [basic] */
			offset = ebgetc(fdb);
			restore_pattern(fdb, offset, 3);
			break;

		case 0x1: /* restore 4 bytes pattern [basic] */
			offset = ebgetc(fdb);
			restore_pattern(fdb, offset, 4);
			break;

		case 0x2: /* restore 5 bytes pattern [basic] */
			offset = ebgetc(fdb);
			restore_pattern(fdb, offset, 5);
			break;

		case 0x3: /* restore 3 bytes pattern [funny] */
			offset = 0x100 + ebgetc(fdb);
			restore_pattern(fdb, offset, 3);
			break;

		case 0x4: /* restore 3 or 4 bytes pattern [rotting] */
			offset = 0;
			flag = get_flag(fdb);
			switch(flag){
			case 0:
				length = 3;
				break;
			case 1:
				length = 4;
				break;
			default:
				length = 3;
				offset = (flag << 8) + ebgetc(fdb);
			}
			restore_pattern(fdb, offset, length);
			break;

		case 0x5: /* restore 4 bytes pattern [advanced] */
			offset = get_adv_offset(fdb);
			restore_pattern(fdb, offset, 4);
			break;

		case 0x6: /* restore 5 bytes pattern [advanced] */
			offset = get_adv_offset(fdb);
			restore_pattern(fdb, offset, 5);
			break;

		case 0x7: /* restore 6 bytes pattern, neads 1 bit subflag */
			if(get_subflag(fdb, 1)){ /* [advanced] */
				offset = get_adv_offset(fdb);
			}else{ /* [basic] */
				offset = ebgetc(fdb);
			}
			restore_pattern(fdb, offset, 6);

			break;

		case 0x8: /* restore 7 or 8 bytes pattern, neads 2 bits subflag */
			subflag = get_subflag(fdb, 2);
			if(subflag & 2){ /* [advanced] */
				offset = get_adv_offset(fdb);
			}else{ /* [basic] */
				offset = ebgetc(fdb);
			}
			length = 7 + (subflag & 1);
			restore_pattern(fdb, offset, length);
			break;

		case 0x9: /* restore 9 to 12 bytes pattern, neads 3 bits subflag */
			subflag = get_subflag(fdb, 3);
			if(subflag & 4){ /* [advanced] */
				offset = get_adv_offset(fdb);
			}else{ /* [basic] */
				offset = ebgetc(fdb);
			}
			switch(subflag & 3){
			case 0: /* 9 bytes */
				restore_pattern(fdb, offset, 9);
				break;

			case 1: /* 11 bytes */
				restore_pattern(fdb, offset, 11);
				break;

			case 2: /* 10 bytes */
				restore_pattern(fdb, offset, 10);
				break;

			case 3: /* 12 bytes */
				restore_pattern(fdb, offset, 12);
				break;
			}
			break;

		case 0xA: /* 1 byte original data */
			store_original(fdb, 1);
			break;

		case 0xB: /* 2 bytes original data */
			store_original(fdb, 2);
			break;

		case 0xC: /* 3 or 4 bytes original data, neads 1 bit subflag */
			length = 3 + get_subflag(fdb, 1);
			store_original(fdb, length);
			break;

		case 0xD: /* 5 to 8 bytes original data, neads 2 bit subflag */
			subflag = get_subflag(fdb, 2);
			switch(subflag){
			case 0: /* 5 bytes */
				store_original(fdb, 5);
				break;

			case 1: /* 7 bytes */
				store_original(fdb, 7);
				break;

			case 2: /* 6 bytes */
				store_original(fdb, 6);
				break;

			case 3: /* 8 bytes */
				store_original(fdb, 8);
				break;
			}
			break;

		case 0xE: /* 9 to 24 bytes original data, neads 1 unit flag */
			length = 9 + get_flag(fdb);
			store_original(fdb, length);
			break;

		case 0xF: /* neads 1 bit subflag */
			if(get_subflag(fdb, 1)){ /* neads more 1 bit subflag */
				if(get_subflag(fdb, 1)){
					/* restore 29 to 284 bytes pattern,
					   neads 1 byte length and facking more... */
					length = ebgetc(fdb);
					if(length == 0xFF){ /* catch the terminater */
						if(debug != NULL){
							fprintf(debug, "[DEBUG] Catch the terminater\n");
						}
						out->data = fdb->data;
						out->num_of_block = fdb->db_next / out->block_size;
						free(fdb->dict.index);
						free(fdb);
						return out->num_of_block;
					}else{
						length += 29;
					}
					if(get_subflag(fdb, 1)){ /* [advanced] */
						offset = get_adv_offset(fdb);
					}else{ /* [basic] */
						offset = ebgetc(fdb);
					}
					restore_pattern(fdb, offset, length);
				}else{ /* 25 to 280 bytes original data */
					length = 25 + ebgetc(fdb);
					store_original(fdb, length);
				}
			}else{ /* restore 13 to 28 bytes pattern,
			          neads 1 unit flag and more 1 bit subflag */
				length = 13 + get_flag(fdb);
				if(get_subflag(fdb, 1)){
					offset = get_adv_offset(fdb);
				}else{
					offset = ebgetc(fdb);
				}
				restore_pattern(fdb, offset, length);
			}
			break;
		}
	}

	out->data = fdb->data;
	out->num_of_block = fdb->db_next / out->block_size;
	free(fdb->dict.index);
	free(fdb->eb);
	free(fdb);

	return 0;
}

int fcd_hi_unit_decode(FCD *in, cd_image_buffer *out, FILE *debug)
{
	size_t n;

	int db_len;
	short eb_len;

	char *eb;
	
	db_len = out->block_size * 16;
	out->data = (char *)malloc(db_len);
	
	read_le_int16(in->stream, &(eb_len));
	in->pos += 2;

	if(eb_len & 0x8000){ /* unit is compressed */
		eb_len &= 0x7FFF;
		eb = (char *)malloc(eb_len);
		if((n = fread(eb, 1, eb_len, in->stream)) != eb_len){
			in->pos += n;
			free(eb);
			goto FCD_HI_UNIT_DECODE_ERROR;
		}
		in->pos += n;
	}else{ /* unit is uncompressed */
		if(eb_len != 0x7FFF){

			/*********************************************************
			              Fucking FCD spec infomation

			    If a unit length is set to 0x7FFF, the correct
			    value is 0x8000. (already setting, don't touch)

			    In the other case, unit length is a correct value.
			    (probably, I have not seem the case yet)
			*********************************************************/

			db_len = eb_len;
		}
		if((n = fread(out->data, 1, db_len, in->stream)) != db_len){
			in->pos += n;
			goto FCD_HI_UNIT_DECODE_ERROR;
		}
		in->pos += n;
		out->num_of_block = n / out->block_size;
		return out->num_of_block;
	}

	if((n = fcd_HiDecompress(out->data, eb, eb_len)) == 0){
		free(eb);
		goto FCD_HI_UNIT_DECODE_ERROR;
	}
	
	free(eb);
	out->num_of_block = n / out->block_size;
	return out->num_of_block;

FCD_HI_UNIT_DECODE_ERROR:
	out->num_of_block = 0;
	return 0;
}
	
static int get_flag(fcd_decoding_buffer *fdb)
{
	int flag;
	
	if(fdb->f.rest == 0){
		if(fdb->eb_pos + 4 > fdb->eb_len){
            return -1;
        }

        fdb->f.data = le_char_array_to_int32(fdb->eb + fdb->eb_pos);
        fdb->eb_pos += 4;
        
        fdb->f.rest = 8;
	}
	
	flag = fdb->f.data & 0xF;
	
	fdb->f.data >>= 4;
	fdb->f.rest -= 1;

	if(fdb->debug != NULL){
		fprintf(fdb->debug, "[DEBUG] Flag: %1X, Rest: ", flag);
		print_flag(fdb->f.data, fdb->f.rest, fdb->debug);
		fprintf(fdb->debug, "\n");
		fflush(fdb->debug);
	}
	
	return flag;
}

static int get_subflag(fcd_decoding_buffer *fdb, int bits)
{
	int subflag;
	int buffer;
	static const int bitmask[] = {
		0x00, 0x01, 0x03, 0x07, 0x0F, 0x1F, 0x3F, 0x7F, 0xFF
	};
	
	if(fdb->sf.rest < bits){
		if((fdb->eb_pos + 4) > fdb->eb_len){
            return -1;
        }
        
        buffer = le_char_array_to_int32(fdb->eb + fdb->eb_pos);
        fdb->eb_pos += 4;
		
		subflag = fdb->sf.data & bitmask[fdb->sf.rest];
		subflag += ((buffer & bitmask[bits - fdb->sf.rest]) << fdb->sf.rest);
		
		fdb->sf.data = buffer >> (bits - fdb->sf.rest);
		fdb->sf.rest = 32 - (bits - fdb->sf.rest);
	}else{
		subflag = fdb->sf.data & bitmask[bits];
		fdb->sf.data >>= bits;
		fdb->sf.rest -= bits;
	}

	if(fdb->debug != NULL){
		fprintf(fdb->debug, "[DEBUG] SubF: ");
		print_bit(subflag, bits, fdb->debug);
		fprintf(fdb->debug, " Rest: ");
		print_bit(fdb->sf.data, fdb->sf.rest, fdb->debug);
		fprintf(fdb->debug, "\n");
		fflush(fdb->debug);
	}

	return subflag;
}

static int get_adv_offset(fcd_decoding_buffer *fdb)
{
	int flag;

	int r; /* return value */

	flag = get_flag(fdb);

	if(flag){
		r = (flag << 8);
		r += ebgetc(fdb);
	}else{
		r = 0x1000;
		r += (get_flag(fdb) << 8);
		r += ebgetc(fdb);
	}

	return r;
}

static int store_original(fcd_decoding_buffer *fdb, int length)
{
	int i;

	if(fdb->debug != NULL){
		fprintf(fdb->debug, 
			"[DEBUG] O-Len: %d, Index: %d, Byte: %d\n\t", 
			length, fdb->dict.last + 1, fdb->db_next);
	}
	
    if((fdb->eb_pos + length) > fdb->eb_len){
        if(fdb->debug != NULL){
            fprintf(fdb->debug, "\n[ERROR] Unexpected End of Buffer\n");
        }
        return 0;
    }
    
    if((fdb->db_next + length) > fdb->db_len){
        if(fdb->debug != NULL){
            fprintf(fdb->debug, "[ERROR] Length over the decoding buffer.\n"); 
        }
        return 0;
    }
    
    for(i=0;i<length;i++){
		fdb->dict.last += 1;
		fdb->dict.index[fdb->dict.last] = fdb->data + fdb->db_next;
		fdb->data[fdb->db_next] = *(fdb->eb + fdb->eb_pos);
		fdb->db_next += 1;
		fdb->eb_pos += 1;
	}
    
	if(fdb->debug != NULL){
		for(i=0;i<length;i++){
			fprintf(fdb->debug, "%02x ", fdb->data[fdb->db_next - length + i] & 0xff);
		}
		fprintf(fdb->debug, "\n");
		fflush(fdb->debug);
	}

	return 1;
}

static int restore_pattern(fcd_decoding_buffer *fdb, int offset, int length)
{
	int i;
	char *data;

	if(fdb->debug != NULL){
		fprintf(fdb->debug, 
			"[DEBUG] P-Len: %d, Offset: %d(%04X), Index: %d. Byte: %d\n\t", 
			length, offset, offset & 0xFFFF,  fdb->dict.last + 1, fdb->db_next);
	}

	
	if((fdb->dict.last - offset < 0) || (offset < 0)){
		if(fdb->debug != NULL){
			fprintf(fdb->debug, "[ERROR] Offset is out of range.\n");
		}
		return 0;
	}
	
	data = fdb->dict.index[fdb->dict.last - offset];
	fdb->dict.last += 1;
	fdb->dict.index[fdb->dict.last] = fdb->data + fdb->db_next;

    if((fdb->db_next + length) > fdb->db_len){
        if(fdb->debug != NULL){
            fprintf(fdb->debug, "[ERROR] Length over the decoding buffer.\n"); 
        }
        return 0;
    }
    
	for(i=0;i<length;i++){
		fdb->data[fdb->db_next] = data[i];
		fdb->db_next += 1;

	}

	if(fdb->debug != NULL){
		for(i=0;i<length;i++){
			fprintf(fdb->debug, "%02X ", data[i] & 0xFF);
		}
		fprintf(fdb->debug, "\n");
		fflush(fdb->debug);
	}
	
	return 1;
}

static void print_flag(int flag, int column, FILE *out)
{
	char format[9];
	
	static const int bitmask[] = {
		0x00000000, 
		0x0000000F, 0x000000FF, 0x00000FFF, 0x0000FFFF, 
		0x000FFFFF, 0x00FFFFFF, 0x0FFFFFFF, 0xFFFFFFFF,
	};
	
	if(column){
		sprintf(format, "0x%%0%dX", column);
		fprintf(out, format, flag & bitmask[column]);
	}else{
		fprintf(out, "none");
	}
}

static int ebgetc(fcd_decoding_buffer *fdb)
{
    int r;

    if(fdb->eb_pos < fdb->eb_len){
        r = (unsigned char)fdb->eb[fdb->eb_pos];
        fdb->eb_pos += 1;
        return r;
    }else{
        return 0;
    }
}